function transform = get_interpolated_transform(x, y_real, y_imag, dz, num_points)
% --------------------------------------------------------------------------------------------------
% Hybrid Matlab/C implementation of cubic spline interpolation in order to achieve better speed.
% --------------------------------------------------------------------------------------------------
% x_real        ... x-coordinates where log-transform is known
% y_real        ... real-part of log-transform at x-coordinates
% y_imag        ... imag-part of log-transform at x-coordinates
% dz            ... spacing of grid for which interpolation is desired
% num_points    ... length of desired grid: 0, dz, 2*dz, ..., (N-1)*dz
% --------------------------------------------------------------------------------------------------
% sample call: cubic_interpolation_matlab_c((0:5), [1 1.5 2 2 1.5 1.4], [6 1 5 2 4 3], 1, 100)
% --------------------------------------------------------------------------------------------------

% Set up the sparse, tridiagonal, linear system b = ?*c for the slopes
n = length(x);
dx = diff(x);
divdif_real = diff(y_real) ./ dx;
divdif_imag = diff(y_imag) ./ dx;
b_real = zeros(1,n);
b_imag = b_real;
b_real(1,2:n-1) = 3*(dx(1,2:n-1) .* divdif_real(1,1:n-2) + dx(1,1:n-2) .* divdif_real(1,2:n-1));
b_imag(1,2:n-1) = 3*(dx(1,2:n-1) .* divdif_imag(1,1:n-2) + dx(1,1:n-2) .* divdif_imag(1,2:n-1));
% Calculate end slopes
x31 = x(3)-x(1);
xn = x(n)-x(n-2);
b_real(1,1) = ((dx(1)+2*x31) * dx(2) * divdif_real(1,1) + dx(1)^2 * divdif_real(1,2)) /x31;
b_real(1,n) = (dx(n-1)^2 * divdif_real(1,n-2) + (2*xn+dx(n-1)) * dx(n-2) * divdif_real(1,n-1)) / xn;
b_imag(1,1) = ((dx(1)+2*x31) * dx(2) * divdif_imag(1,1) + dx(1)^2 * divdif_imag(1,2)) /x31;
b_imag(1,n) = (dx(n-1)^2 * divdif_imag(1,n-2) + (2*xn+dx(n-1)) * dx(n-2) * divdif_imag(1,n-1)) / xn;
% Sparse linear equation solution for the slopes
dxt = dx(:);
c = spdiags([ [x31;     dxt(1:n-2);                 0] ...
              [dxt(2);  2*(dxt(2:n-1)+dxt(1:n-2));  dxt(n-2)] ...
              [0;       dxt(2:n-1);                 xn] ], [-1 0 1], n, n);
s_real = b_real / c;
s_imag = b_imag / c;

% Define cubic spline
pp_real = pwch(x, y_real, s_real, dx, divdif_real);
pp_real.dim = 1;
pp_imag = pwch(x, y_imag, s_imag, dx, divdif_imag);
pp_imag.dim = 1;

%%%%%%%%%%%%%%%%%%%%%%%%%
% Evaluate cubic spline %
%%%%%%%%%%%%%%%%%%%%%%%%%

% Take apart PP
[b_real, c_real] = unmkpp(pp_real);
[b_imag, c_imag] = unmkpp(pp_imag);

% C implementation
if (0)
    % Hyprid C/Matlab => lots of internal overhead
    interpolated_real = evaluate_cubic_spline_c2(dz, num_points, b_real, c_real);
    interpolated_imag = evaluate_cubic_spline_c2(dz, num_points, b_imag, c_imag);
    transform = exp(complex(interpolated_real, interpolated_imag));
else
    % Pure C implementation
    transform = interp_and_exp_transform_c(dz, num_points, b_real, c_real, b_imag, c_imag);
end




